--author: igor29381

RailRoad = {};
RailRoad_mt = Class(RailRoad, Object);

RailRoad.trainNameToInt = {["vl80a"]=1, ["vl80b"]=2, ["ep1m"]=3, ["tem7"]=4, ["poluvagon"]=5, ["boxcar"]=6, ["hopper"]=7, ["platforma"]=8, ["tanker"]=9, ["pass"]=10};
RailRoad.trainIntToName = {"vl80a", "vl80b", "ep1m", "tem7", "poluvagon", "boxcar", "hopper", "platforma", "tanker", "pass"};

function RailRoad.onCreate(id)
    local object = RailRoad:new(g_server ~= nil, g_client ~= nil)
    if object:load(id) then
        g_currentMission:addOnCreateLoadedObject(object)
        object:register(true)
    else
        object:delete()
    end;
end;

function RailRoad:new(isServer, isClient, customMt)
    local mt = customMt
    if mt == nil then
        mt = RailRoad_mt
    end;
    local self = Object:new(isServer, isClient, RailRoad_mt);
	self.railRoadDirtyFlag = self:getNextDirtyFlag();
    return self;
end;

function RailRoad:load(id)
	TrafficManager.railRoadId = self.id;
	local splinesRootNode = getChild(id, "splines");
	local crossingsRootNode = getChild(id, "crossings");
	self.staticCarsRootNode = getChild(id, "staticCars");
	self.splines = {};
	self.crossings = {};
	self.staticCars = {};
	self.numStaticCars = {};
	self.train = {};
	local actions = {	["horn"]=true,
						["whistle"]=true,
						["stationSpeed"]=true,
						["trackSpeed"]=true,
						["stop"]=true,
						["crossing"]=true};
	if splinesRootNode > 0 then
		local numSplines = getNumOfChildren(splinesRootNode);
		if numSplines > 0 then
			for i=1, numSplines do
				local splineId = getChildAt(splinesRootNode, i-1);
				if splineId ~= nil then
					local spline = {};
					spline.nodeId = splineId;
					spline.length = getSplineLength(splineId);
					local numMarkers = getNumOfChildren(splineId);
					if numMarkers > 0 then
						spline.markers = {};
						for m=1, numMarkers do
							local markerId = getChildAt(splineId, m-1);
							if markerId ~= nil then
								local action = getUserAttribute(markerId, "action");
								if action and actions[action] then
									local x, _, z = getWorldTranslation(markerId);
									local marker = {["x"]=x, ["z"]=z, ["action"]=action};
									local minutes = Utils.getNoNil(getUserAttribute(markerId, "minutes"), 60)*60;
									local stationName = Utils.getNoNil(getUserAttribute(markerId, "stationName"), "stationName");
									if marker.action == "stop" then
										marker.minutes = minutes;
										marker.stationName = stationName;
									end;
									table.insert(spline.markers, marker);
								end;
							end;
						end;
					end;
					table.insert(self.splines, spline);
				end;
			end;
		end;
	end;
	if crossingsRootNode > 0 then
		local numCrossings = getNumOfChildren(crossingsRootNode);
		if numCrossings > 0 then
			for i=1, numCrossings do
				local crossingId = getChildAt(crossingsRootNode, i-1);
				if crossingId ~= nil then
					local crossing = RailRoadCrossing:new(g_server ~= nil, g_client ~= nil);
					g_currentMission:addOnCreateLoadedObject(crossing);
					crossing:load(crossingId);
					crossing:register(true);
					crossing.parent = self;
					table.insert(self.crossings, crossing);
				end;
			end;
		end;
	end;
	if self.isServer and self.staticCarsRootNode > 0 then
		local num = getNumOfChildren(self.staticCarsRootNode);
		if num > 0 then
			for i=1, num do
				local groupId = getChildAt(self.staticCarsRootNode, i-1);
				if groupId ~= nil then
					local numPoints = getNumOfChildren(groupId);
					if numPoints > 0 then
						numPoints = math.floor(math.random(numPoints/2, numPoints));
						table.insert(self.numStaticCars, numPoints);
						self:loadStaticCars(groupId, numPoints);
					end;
				end;
			end;
		end;
	end;
	self.cars = {};
	self.carTypes = {};
	self.isEnabled = true;
	self.trainOnTrack = false;
	self.trainIsStopped = false;
	self.enableHorn = false;
	self.enableClientHorn = false;
	self.enableWhistle = false;
	self.enableClientWhistle = false;
	self.onStation = false;
	self.onWoodStage = false;
	self.showPenaltyMessage = false;
	self.trainWasStoppedOnTarget = false;
	self.spl = 0;
	self.limitedSpeed = 0;
	self.penaltyTimer = 0;
	self.hornTimer = 0;
	self.brakeTimer = 0;
	self.stopTimer = 0;
	self.splinePosition = 0;
	self.controlSplinePosition = 0;
	self.brakeCoeff = 2;
	self.moneyChangeId = getMoneyTypeId();
	self.lastMoneyChange = -1;
	g_currentMission.environment:addWeatherChangeListener(self);
	g_currentMission.environment:addHourChangeListener(self);
	UniversalFactoryHUD.RailRoad = self;
    return true;
end;

function RailRoad:delete()
	if #self.cars > 0 then
		for i=1, #self.cars do
			delete(self.cars[i]);
		end;
	end;
	for _,car in pairs(self.staticCars) do
		delete(car);
	end;
	if g_currentMission.environment ~= nil then
		g_currentMission.environment:removeWeatherChangeListener(self);
		g_currentMission.environment:removeHourChangeListener(self);
	end;
	RailRoad:superClass().delete(self);
end;

function RailRoad:readStream(streamId, connection)
	RailRoad:superClass().readStream(self, streamId);
	if connection:getIsServer() then
		if self.staticCarsRootNode > 0 then
			local num = getNumOfChildren(self.staticCarsRootNode);
			if num > 0 then
				for i=1, num do
					local groupId = getChildAt(self.staticCarsRootNode, i-1);
					if groupId ~= nil then
						local numPoints = streamReadInt8(streamId);
						self:loadStaticCars(groupId, numPoints);
					end;
				end;
			end;
		end;
		self.onStation = streamReadBool(streamId);
		self.onWoodStage = streamReadBool(streamId);
		local numCars = streamReadInt8(streamId);
		local splineIndex = streamReadInt8(streamId);
		self.limitedSpeed = streamReadInt8(streamId);
		if numCars > 0 and splineIndex > 0 then
			for i=1, numCars do
				local typeCar = streamReadInt8(streamId);
				table.insert(self.carTypes, typeCar);
			end;
			self:loadNewTrain(splineIndex, self.limitedSpeed);
		end;
		self.splinePosition = streamReadInt16(streamId)/1000;
	end;
end;

function RailRoad:writeStream(streamId, connection)
	RailRoad:superClass().writeStream(self, streamId);
	if not connection:getIsServer() then
		if self.staticCarsRootNode > 0 then
			local num = getNumOfChildren(self.staticCarsRootNode);
			if num > 0 then
				for i=1, num do
					streamWriteInt8(streamId, self.numStaticCars[i]);
				end;
			end;
		end;
		local numCars = #self.carTypes;
		streamWriteBool(streamId, self.onStation);
		streamWriteBool(streamId, self.onWoodStage);
		streamWriteInt8(streamId, numCars);
		streamWriteInt8(streamId, self.spl);
		streamWriteInt8(streamId, self.limitedSpeed);
		if #self.carTypes > 0 then
			for i=1, #self.carTypes do
				streamWriteInt8(streamId, self.carTypes[i]);
			end;
		end;
		streamWriteInt16(streamId, math.floor(self.splinePosition*1000));
	end;
end;

function RailRoad:readUpdateStream(streamId, timestamp, connection)
	RailRoad:superClass().readUpdateStream(self, streamId, timestamp, connection);
	if connection:getIsServer() then
		self.controlSplinePosition = streamReadInt32(streamId)/1000;
		self.limitedSpeed = streamReadInt32(streamId)/100;
		local distance = (self.controlSplinePosition-self.splinePosition)*self.splines[self.spl].length;
		self.limitedSpeed = Utils.clamp(self.limitedSpeed+distance/3, math.max(self.limitedSpeed-5, 0), self.limitedSpeed+10);
		local trainIsStopped = streamReadBool(streamId);
		if self.trainIsStopped and not trainIsStopped and self.onStation then
			for _,car in pairs(self.cars) do
				self:randomPlaneMaterial(car);
			end;
		end;
		self.trainIsStopped = trainIsStopped;
		if self.trainIsStopped then
			setVisibility(TrafficManager.brakeSound, false);
		end;
		self.brakeCoeff = streamReadInt8(streamId);
		self.enableHorn = streamReadBool(streamId);
		self.enableWhistle = streamReadBool(streamId);
		self.trainWasStoppedOnTarget = streamReadBool(streamId);
		local showPenaltyMessage = streamReadBool(streamId);
		if showPenaltyMessage then
			g_currentMission:showBlinkingWarning(g_i18n:getText("train_standing"), 5000);
		end;
	end;
end;

function RailRoad:writeUpdateStream(streamId, connection, dirtyMask)
	RailRoad:superClass().writeUpdateStream(self, streamId, connection, dirtyMask);
	if not connection:getIsServer() then
		streamWriteInt32(streamId, math.floor(self.controlSplinePosition*1000));
		streamWriteInt32(streamId, math.floor(self.limitedSpeed*100));
		streamWriteBool(streamId, self.trainIsStopped);
		streamWriteInt8(streamId, self.brakeCoeff);
		streamWriteBool(streamId, self.enableClientHorn);
		self.enableClientHorn = false;
		streamWriteBool(streamId, self.enableClientWhistle);
		self.enableClientWhistle = false;
		streamWriteBool(streamId, self.trainWasStoppedOnTarget);
		streamWriteBool(streamId, self.showPenaltyMessage);
		self.showPenaltyMessage = false;
	end;
end;

function RailRoad:loadStaticCars(groupId, numPoints)
	local trainzRoot = Utils.loadSharedI3DFile(TrafficManager.trainsI3dFilename);
	local node = getChildAt(trainzRoot, 0);
	if getNumOfChildren(node) > 0 then
		for p=1, numPoints do
			local point = getChildAt(groupId, p-1);
			if point ~= nil then
				local x, y, z = getWorldTranslation(point);
				local rx, ry, rz = getWorldRotation(point);
				local typeCar = getUserAttribute(point, "typeCar");
				car = clone(getChild(node, typeCar), false);
				link(getRootNode(), car);
				self:loadCollision(car);
				self:randomPlaneMaterial(car);
				if TrafficManager.materials[typeCar] then
					local randomMat = math.random(1, #TrafficManager.materials[typeCar]);
					setMaterial(car, TrafficManager.materials[typeCar][randomMat], 0);
				end;
				setTranslation(car, x, y, z);
				setRotation(car, rx, ry, rz);
				table.insert(self.staticCars, car);
			end;
		end;
	end;
	delete(trainzRoot);
end;

function RailRoad:weatherChanged()
	if g_currentMission and g_currentMission.environment then
		if self.nightLights then
			for i=1, #self.nightLights do
				setVisibility(self.nightLights[i], not g_currentMission.environment.isSunOn);
			end;
		end;
	end;
end;

function RailRoad:update(dt)
	if self.isEnabled then
		if self.trainOnTrack then
			if self.splinePosition < 1 and self.splinePosition > 0 then
				local spline = self.splines[self.spl];
				local x,y,z = getSplinePosition(spline.nodeId, self.splinePosition);
				if not self.trainIsStopped then
					if self.isServer then
						self.limitedSpeed = self.currentPartSpeed;
						if self.controlSplinePosition + 0.01 < self.splinePosition then
							self.controlSplinePosition = self.splinePosition;
							self:raiseDirtyFlags(self.railRoadDirtyFlag);
						end;
					end;
					local timeScale = (self.speed/3.6) / spline.length;
					self.splinePosition = self.splinePosition + 0.001*dt*timeScale;
					setTranslation(self.cars[1], x, y, z);
				end;
				if self.isServer then
					if not self.trainIsStopped then
						local targets = {};
						for i=1, #g_currentMission.vehicles do
							targets[i] = g_currentMission.vehicles[i].rootNode;
						end;
						for k,player in pairs(g_currentMission.players) do
							if player.isControlled then
								table.insert(targets, player.rootNode);
							end;
						end;
						self.brakeCoeff = 2;
						local targetFound = false;
						for i=1, #targets do
							local vx, vy, vz = getWorldTranslation(targets[i]);
							local distance = Utils.vector3Length(x-vx, y-vy, z-vz);
							local lx, ly, lz = worldToLocal(self.cars[1], vx, vy, vz);
							if distance < 200 and lz < 0 and ly < 4 and ly > -1 and math.abs(lx) < 15 then
								local position = self.splinePosition + (distance/spline.length);
								local px,py,pz = getSplinePosition(spline.nodeId, position);
								local disToTrack = Utils.vector3Length(px-vx, py-vy, pz-vz);
								if disToTrack < 2 then
									self.enableHorn = true;
									self.limitedSpeed = math.min((distance-12)/3, self.speed);
									self.brakeCoeff = 5;
									if self.speed < 1 then
										self.limitedSpeed = 0;
										targetFound = true;
										local money = -dt*0.1;
										g_currentMission:addSharedMoney(money, "other");
										g_currentMission:addMoneyChange(money, self.moneyChangeId);
										self.lastMoneyChange = 30;
										self.penaltyTimer = self.penaltyTimer + dt;
										if self.penaltyTimer > 60000 then
											self.penaltyTimer = 0;
											if self:notCPVehicle(targets[i]) then
												Economica.statistics.railroadPenalty = Economica.statistics.railroadPenalty + 1;
											end;
										end;
										if not self.trainWasStoppedOnTarget then
											if self:notCPVehicle(targets[i]) then
												Economica.statistics.railroadPenalty = Economica.statistics.railroadPenalty + 1;
											end;
											self.penaltyTimer = 0;
											g_currentMission:showBlinkingWarning(g_i18n:getText("train_standing"), 5000);
											self.showPenaltyMessage = true;
											self.trainWasStoppedOnTarget = true;
											self:raiseDirtyFlags(self.railRoadDirtyFlag);
										end;
									end;
								end;
							end;
						end;
						if not targetFound and self.trainWasStoppedOnTarget then
							self.trainWasStoppedOnTarget = false;
						end;
					end;
					if spline.markers then
						local marker = spline.markers[self.curMark];
						local markerDistance = Utils.vector2Length(x-marker.x, z-marker.z);
						if marker.action == "stop" then
							if self.trainIsStopped then
								if marker.stationName then
									local station = UniversalFactory[marker.stationName];
									if station then
										if self.stopTimer == 0 and self.onStation then
											station.cargoFillTypes = self.trainFillTypes;
											station.cargoLevel = 0;
										end;
										if UniversalFactoryHUD.settings[UniversalFactoryHUD.SET_RANDOMEXTRAMISSION] then
											if self.stopTimer == 0 and self.onStation then
												if Economica.missionInterval < 3 then
													local enableMission = math.random(1, 2) > 1;
													if enableMission then
														self.missionTime = math.random(6, 10)*60;
													end;
												end;
											end;
											if self.missionTime and self.missionTime > marker.minutes - self.stopTimer then
												local fillType = station:getFillTypeWithMinLevel();
												if fillType > 0 then
													Economica:newExtraMission(marker.stationName, fillType, self.missionTime - 60);
												end;
												self.missionTime = nil;
											end;
										end;
										if self.onStation and station.cargoFillTypes then
											for n,ft in pairs(station.fillType) do
												if station.cargoFillTypes[n] and ft.level > 0 then
													self.level = math.min(self.level + 0.25*dt*g_currentMission.missionInfo.timeScale, self.capacity);
												end;
											end;
											if self.level == self.capacity then
												station.cargoFillTypes = nil;
											end;
										end;
									end;
								end;
								self.stopTimer = self.stopTimer + dt*0.001*g_currentMission.missionInfo.timeScale;
							else
								if markerDistance > 2 then
									self.limitedSpeed = math.max(math.min(markerDistance/3, math.min(self.speed, self.currentPartSpeed)), 4);
								else
									self.limitedSpeed = 0;
									self.brakeCoeff = 4;
									if self.speed == 0 then
										self.trainIsStopped = true;
										setVisibility(TrafficManager.brakeSound, false);
										self:raiseDirtyFlags(self.railRoadDirtyFlag);
									end;
								end;
								if markerDistance < 30 and self.controlSplinePosition + 0.0005 < self.splinePosition then
									self.controlSplinePosition = self.splinePosition;
									self:raiseDirtyFlags(self.railRoadDirtyFlag);
								end;
							end;
							if self.stopTimer > marker.minutes then
								if marker.stationName then
									local station = UniversalFactory[marker.stationName];
									if station and self.onStation then
										station.cargoFillTypes = nil;
										for _,car in pairs(self.cars) do
											self:randomPlaneMaterial(car);
										end;
									end;
								end;
								self.stopTimer = 0;
								self.curMark = math.min(self.curMark + 1, #spline.markers);
								self.trainIsStopped = false;
								self.brakeCoeff = 2;
								self.currentPartSpeed = 15;
								self.enableHorn = true;
							end;
						elseif not self.trainIsStopped and markerDistance < 1 then
							if marker.action == "horn" then self.enableHorn = true; end;
							if marker.action == "whistle" then self.enableWhistle = true; end;
							if marker.action == "stationSpeed" then self.currentPartSpeed = 15; end;
							if marker.action == "trackSpeed" then self.currentPartSpeed = self.nominalSpeed; end;
							if marker.action == "crossing" then
								local dis = 5000;
								local nearestCrossing = 0;
								for c=1, #self.crossings do
									local crossing = self.crossings[c];
									local tx, _, tz = getWorldTranslation(self.cars[1]);
									local disToCrossing = Utils.vector2Length(tx-crossing.x, tz-crossing.z);
									if disToCrossing < dis then
										dis = disToCrossing;
										nearestCrossing = c;
									end;
								end;
								if nearestCrossing > 0 then
									local crossing = self.crossings[nearestCrossing];
									crossing.currentTrain = self.cars[1];
									crossing.trainLength = self.trainLength;
									crossing.isClosed = true;
									crossing:raiseDirtyFlags(crossing.crossingDirtyFlag);
								end;
							end;
							self.curMark = math.min(self.curMark + 1, #spline.markers);
						end;
					end;
					if self.lastMoneyChange > 0 then
						self.lastMoneyChange = self.lastMoneyChange - 1;
						if self.lastMoneyChange == 0 then
							g_currentMission:showMoneyChange(self.moneyChangeId, g_i18n:getText("penalties"));
						end;
					end;
				end;
				if self.speed < self.limitedSpeed then
					self.speed = math.min(self.speed + dt*0.0015, self.limitedSpeed);
				else
					self.speed = math.max(self.speed - dt*0.0015*self.brakeCoeff, self.limitedSpeed);
				end;
				setSamplePitch(self.clatterSample, math.min(math.max(self.speed/30, 0.7), 1.3));
				if self.speed > 1 then
					if not getVisibility(self.clatterSound) then
						setVisibility(self.clatterSound, true);
					end;
				else
					if getVisibility(self.clatterSound) then
						setVisibility(self.clatterSound, false);
					end;
				end;
				if not self.trainIsStopped then
					if self.brakeCoeff > 2 and not self.trainWasStoppedOnTarget then
						if not getVisibility(TrafficManager.brakeSound) then
							setVisibility(TrafficManager.brakeSound, true);
						end;
						if self.speed < 1 and not getVisibility(TrafficManager.brakeEndSound) then
							setVisibility(TrafficManager.brakeSound, false);
							setVisibility(TrafficManager.brakeEndSound, true);
							self.brakeTimer = 5000;
						end;
					elseif getVisibility(TrafficManager.brakeSound) then
						setVisibility(TrafficManager.brakeSound, false);
					end;
					for _, wheel in pairs (self.wheels) do
						local wx, _, _ = getRotation(wheel);
						local delta = (self.splinePosition-self.lastSP)*spline.length;
						delta = math.atan(delta/0.5);
						setRotation(wheel, wx-delta, 0, 0);
					end;
					local cam = getCamera();
					local cx, cy, cz = getWorldTranslation(cam);
					local distance = 5000;
					for i=1, #self.cars do
						local car = self.cars[i];
						local position = self.splinePosition - self.lengths[i];
						local px, py, pz = getSplinePosition(spline.nodeId, position);
						if i>1 then
							local carx, cary, carz = getWorldTranslation(self.attachers[i]);
							setTranslation(car, carx, cary, carz);
						end;
						local carx, cary, carz = getWorldTranslation(car);
						local dis = Utils.vector3Length(carx-cx, cary-cy, carz-cz);
						if dis < distance then
							distance = dis;
							setTranslation(self.clatterSound, carx, cary, carz);
							if getVisibility(TrafficManager.brakeSound) then
								setTranslation(TrafficManager.brakeSound, carx, cary, carz);
							end;
							if getVisibility(TrafficManager.brakeEndSound) then
								setTranslation(TrafficManager.brakeEndSound, carx, cary, carz);
							end;
						end;
						local _, ry, _ = getRotation(car);
						local dx, dy, dz = worldToLocal(car, px, py, pz);
						setRotation(car, 0, ry+(0.05*dx), 0);
					end;
				end;
				if self.enableHorn and self.hornTimer == 0 then
					setTranslation(TrafficManager.hornSound, x, y, z);
					setVisibility(TrafficManager.hornSound, true);
					self.hornTimer = 2000;
					if self.isServer then
						self.enableClientHorn = true;
						self:raiseDirtyFlags(self.railRoadDirtyFlag);
					end;
					self.enableHorn = false;
				end;
				if self.enableWhistle and self.hornTimer == 0 then
					setTranslation(TrafficManager.whistleSound, x, y, z);
					setVisibility(TrafficManager.whistleSound, true);
					self.hornTimer = 3500;
					if self.isServer then
						self.enableClientWhistle = true;
						self:raiseDirtyFlags(self.railRoadDirtyFlag);
					end;
					self.enableWhistle = false;
				end;
				if self.hornTimer > 0 then
					self.hornTimer = math.max(self.hornTimer - dt, 0);
				end;
				if self.hornTimer == 0 then
					setVisibility(TrafficManager.hornSound, false);
					setVisibility(TrafficManager.whistleSound, false);
				end;
				if self.brakeTimer > 0 then
					self.brakeTimer = math.max(self.brakeTimer - dt, 0);
				end;
				if self.brakeTimer == 0 then
					setVisibility(TrafficManager.brakeEndSound, false);
				end;
				self.lastSP = self.splinePosition;
			else
				self.trainOnTrack = false;
				for i=1, #self.cars do
					delete(self.cars[i]);
				end;
				setVisibility(self.clatterSound, false);
				self.onStation = false;
				self.onWoodStage = false;
				self.nightLights = nil;
				self.cars = {};
			end;
		end;
	end;
end;

function RailRoad:hourChanged()
	if self.isServer and not self.trainOnTrack then
		if UniversalFactoryHUD.settings[UniversalFactoryHUD.SET_RANDOMEVENTS] and Economica.randomEventsByNames.railRoadDamage.active then
			return;
		end;
		for tt=1, #TrafficManager.trainsTimetable do
			local timetable = TrafficManager.trainsTimetable[tt];
			if timetable.hour == g_currentMission.environment.currentHour then
				self.carTypes = {};
				local train = RailRoad.trainNameToInt[timetable.train];
				self.onStation = timetable.onStation;
				self.onWoodStage = timetable.onWoodStage;
				table.insert(self.carTypes, train);
				if train == 1 then
					table.insert(self.carTypes, 2);
					local numTypes = math.random(1, 4);
					for i=1, numTypes do
						local numCars = math.random(math.ceil(20/numTypes), math.ceil(25/numTypes));
						local typeCar = math.random(5, 9);
						for n=1, numCars do
							table.insert(self.carTypes, typeCar);
						end;
					end;
				end;
				if train == 3 then
					for i=1, 6 do table.insert(self.carTypes, 10); end;
				end;
				if train == 4 then
					if self.onWoodStage then
						for i=1, 3 do table.insert(self.carTypes, 5); end;
					elseif timetable.cars then
						local numTypes = #timetable.cars;
						for nt=1, numTypes do
							local numCars = math.random(math.ceil(10/numTypes), math.ceil(12/numTypes));
							for n=1, numCars do
								table.insert(self.carTypes, timetable.cars[nt]);
							end;
						end;
					else
						local numTypes = math.random(1, 3);
						for nt=1, numTypes do
							local numCars = math.random(math.ceil(10/numTypes), math.ceil(12/numTypes));
							local typeCar = math.random(5, 7);
							for n=1, numCars do
								table.insert(self.carTypes, typeCar);
							end;
						end;
					end;
				end;
				self:loadNewTrain(timetable.splineIndex, timetable.speed);
				g_server:broadcastEvent(RailRoadEvent:new(self.carTypes, timetable.splineIndex, timetable.speed, self.onStation, self.onWoodStage, self));
				break;
			end;
		end;
	end;
end;

function RailRoad:loadNewTrain(splineIndex, speed)
	self.cars = {};
	self.lengths = {};
	self.wheels = {};
	self.attachers = {};
	self.trainFillTypes = {};
	self.nightLights = {};
	local trainzRoot = Utils.loadSharedI3DFile(TrafficManager.trainsI3dFilename);
	local node = getChildAt(trainzRoot, 0);
	if getNumOfChildren(node) > 0 then
		for i=1, #self.carTypes do
			local car = clone(getChild(node, RailRoad.trainIntToName[self.carTypes[i]]), false);
			link(getRootNode(), car);
			table.insert(self.cars, car);
		end;
	end;
	delete(trainzRoot);
	self.spl = splineIndex;
	local splineLength = self.splines[splineIndex].length;
	local length = 0;
	self.capacity = (#self.cars - 1)*300000;
	self.level = 0;
	for c=1, #self.cars do
		local car = self.cars[c];
		local name = RailRoad.trainIntToName[self.carTypes[c]];
		self:loadCollision(car);
		if c>1 then
			self.attachers[c] = getChild(self.cars[c-1], "attacher");
			local carx, cary, carz = getWorldTranslation(self.attachers[c]);
			setTranslation(car, carx, cary, carz);
		end;
		local bogey = getChild(car, "bogey");
		local _, _, z = getWorldTranslation(bogey);
		local pos = z/splineLength;
		length = z;
		self.lengths[c] = pos;
		local wheels = getUserAttribute(car, "wheels");
		if wheels then
			local wheelsTable = Utils.splitString(" ", wheels);
			for k=1, #wheelsTable do
				local wheel = Utils.indexToObject(car, wheelsTable[k]);
				if wheel then table.insert(self.wheels, wheel); end;
			end;
		end;
		if self.onStation then
			local cargoTable = {};
			cargoTable.poluvagon = {"potato", "sugarBeet", "wood", "onion", "carrot", "plank"};
			cargoTable.hopper = {"wheat", "barley", "rape", "maize", "sunflower", "rye", "soybean"};
			cargoTable.boxcar = {"packed_seed_oil", "packed_milk", "beer", "sugar", "konditer", "sweetmilk", "canning", "wheatflour", "ryeflour", "maizeflour", "furniture"};
			if cargoTable[name] then
				for _,ft in pairs(cargoTable[name]) do
					local fillType = FillUtil.fillTypeNameToInt[ft];
					if fillType then self.trainFillTypes[fillType] = true; end;
				end;
			end;
		elseif not self.onWoodStage then
			local cargoTable = getUserAttribute(car, "randomCargo");
			if cargoTable then
				local randomCargo = {};
				cargoTable = Utils.splitString(" ", cargoTable);
				for k=1, #cargoTable do
					local cargo = Utils.indexToObject(car, cargoTable[k]);
					if cargo then table.insert(randomCargo, cargo); end;
				end;
				local cargo = math.random(1, #randomCargo);
				setVisibility(randomCargo[cargo], true);
				for k=1, #randomCargo do
					if k~=cargo then delete(randomCargo[k]); end;
				end;
			end;
			self:randomPlaneMaterial(car);
		end;
		if TrafficManager.materials[name] then
			local randomMat = math.random(1, #TrafficManager.materials[name]);
			setMaterial(car, TrafficManager.materials[name][randomMat], 0);
		end;
		local nightLights = getChild(self.cars[c], "nightLights");
		if nightLights > 0 then
			table.insert(self.nightLights, nightLights);
		end;
		for i=1, #self.nightLights do
			setVisibility(self.nightLights[i], not g_currentMission.environment.isSunOn);
		end;
	end;
	self.trainLength = length;
	if self.splines[splineIndex].markers then
		self.curMark = 1;
	end;
	self.nominalSpeed = speed;
	self.limitedSpeed = speed;
	self.currentPartSpeed = speed;
	self.speed = speed;
	if self.carTypes[1] ~= 3 then
		self.clatterSound = TrafficManager.clatterHardSound;
	else
		self.clatterSound = TrafficManager.clatterSound;
	end;
	setVisibility(self.clatterSound, true);
	self.clatterSample = getAudioSourceSample(self.clatterSound);
	self.splinePosition = (self.trainLength+10)/splineLength;
	self.controlSplinePosition = self.splinePosition;
	self.lastSP = self.splinePosition;
	local x,y,z = getSplinePosition(self.splines[splineIndex].nodeId, self.splinePosition);
	local rx,ry,rz = getSplineOrientation(self.splines[splineIndex].nodeId, self.splinePosition, 0, -1, 0);
	setTranslation(self.cars[1], x, y, z);
	for c=1, #self.cars do
		setRotation(self.cars[c], rx,ry,rz);
	end;
	self.trainOnTrack = true;
end;

function RailRoad:loadCollision(car)
	local collision = getUserAttribute(car, "collision");
	if collision then
		collision = TrafficManager.curModDir..collision;
		if fileExists(collision) then
			local collisionRoot = Utils.loadSharedI3DFile(collision);
			local node = getChildAt(collisionRoot, 0);
			link(car, node);
			delete(collisionRoot);
		end;
	end;
end;

function RailRoad:randomPlaneMaterial(car)
	local plane = getChild(car, "plane");
	if plane > 0 and TrafficManager.materials["plane"] then
		local randomMat = math.random(1, #TrafficManager.materials["plane"]);
		setMaterial(plane, TrafficManager.materials["plane"][randomMat], 0);
		setVisibility(plane, true);
	end;
end;

function RailRoad:notCPVehicle(node)
	local vehicle = g_currentMission.nodeToVehicle[node];
	if vehicle and vehicle.cp then
		local rootAttacherVehicle = vehicle:getRootAttacherVehicle();
		if rootAttacherVehicle and rootAttacherVehicle.cp.isDriving then
			return false;
		end;
	end;
	return true;
end;

g_onCreateUtil.addOnCreateFunction("RailRoad", RailRoad.onCreate);

--------------------------------------------------------------------------------------EVENT

RailRoadEvent = {};
RailRoadEvent_mt = Class(RailRoadEvent, Event);

InitEventClass(RailRoadEvent, "RailRoadEvent");

function RailRoadEvent:emptyNew()
	local self = Event:new(RailRoadEvent_mt);
    return self;
end;

function RailRoadEvent:new(carTypes, splineIndex, speed, onStation, onWoodStage, object)
	local self = RailRoadEvent:emptyNew();
	self.carTypes = carTypes;
	self.splineIndex = splineIndex;
	self.speed = speed;
	self.onStation = onStation;
	self.onWoodStage = onWoodStage;
	self.object = object;
	return self;
end;

function RailRoadEvent:readStream(streamId, connection)
	local id = streamReadInt32(streamId);
    local railRoad = networkGetObject(id);
	local splineIndex = streamReadInt8(streamId);
	local speed = streamReadInt8(streamId);
	local onStation = streamReadBool(streamId);
	local onWoodStage = streamReadBool(streamId);
	local numCars = streamReadInt8(streamId);
	if railRoad ~= nil and numCars > 0 then
		railRoad.carTypes = {};
		for i=1, numCars do
			local typeCar = streamReadInt8(streamId);
			table.insert(railRoad.carTypes, typeCar);
		end;
		railRoad.onStation = onStation;
		railRoad.onWoodStage = onWoodStage;
		railRoad:loadNewTrain(splineIndex, speed);
	end;
end;

function RailRoadEvent:writeStream(streamId, connection)
	streamWriteInt32(streamId, networkGetObjectId(self.object));
	streamWriteInt8(streamId, self.splineIndex);
	streamWriteInt8(streamId, self.speed);
	streamWriteBool(streamId, self.onStation);
	streamWriteBool(streamId, self.onWoodStage);
	local numCars = #self.carTypes;
	streamWriteInt8(streamId, numCars);
	if #self.carTypes > 0 then
		for i=1, #self.carTypes do
			streamWriteInt8(streamId, self.carTypes[i]);
		end;
	end;
end;

---------------------------------------------------------------------------------------

RailRoadCrossing = {};
RailRoadCrossing_mt = Class(RailRoadCrossing, Object);

function RailRoadCrossing.onCreate(id)
    local object = RailRoadCrossing:new(g_server ~= nil, g_client ~= nil)
    if object:load(id) then
        g_currentMission:addOnCreateLoadedObject(object)
        object:register(true)
    else
        object:delete()
    end;
end;

function RailRoadCrossing:new(isServer, isClient, customMt)
    local mt = customMt
    if mt == nil then
        mt = RailRoadCrossing_mt
    end;
    local self = Object:new(isServer, isClient, RailRoadCrossing_mt);
	self.crossingDirtyFlag = self:getNextDirtyFlag();
    return self;
end;

function RailRoadCrossing:load(id)
	local barriers = getUserAttribute(id, "barriers");
	if barriers then barriers = Utils.splitString(" ", barriers); end;
	if barriers then
		self.barriers = {};
		for b=1, #barriers do
			local barrierId = Utils.indexToObject(id, barriers[b]);
			if barrierId > 0 then
				local barrier = {};
				barrier.node = barrierId;
				barrier.speed = 0.0002;
				barrier.position = 0;
				barrier.endRotZ = math.rad(87);
				barrier.isClosed = false;
				barrier.opening = false;
				barrier.closing = false;
				table.insert(self.barriers, barrier);
				if OriginalFunctions.courseplay and g_currentMission.nodeToVehicle then
					local pathVehicle = {};
					pathVehicle.rootNode = getChildAt(barrier.node, 0);
					pathVehicle.name = "barrier";
					pathVehicle.sizeLength = 3;
					pathVehicle.sizeWidth = 3;
					g_currentMission.nodeToVehicle[pathVehicle.rootNode] = pathVehicle;
				end;
			end;
		end;
	end;
	local lights = getUserAttribute(id, "lights");
	if lights then lights = Utils.splitString(" ", lights); end;
	if lights then
		self.lights = {};
		for l=1, #lights do
			local lightId = Utils.indexToObject(id, lights[l]);
			if lightId > 0 then
				setVisibility(lightId, false);
				table.insert(self.lights, lightId);
			end;
		end;
	end;
	local ring = getUserAttribute(id, "ring");
	if ring then ring = Utils.indexToObject(id, ring); end;
	if ring then
		self.ring = ring;
		setVisibility(ring, false);
	end;
	if self.isServer then
		local trigger = getUserAttribute(id, "trigger");
		if trigger ~= nil then
			self.trigger = Utils.indexToObject(id, trigger);
			if self.trigger ~= nil then addTrigger(self.trigger, "triggerCallback", self); end;
		end;
		local x, _, z = getWorldTranslation(id);
		self.x = x;
		self.z = z;
		self.lastDistance = 1000;
	end;
	self.isClosed = false;
	self.wasClosed = false;
	self.showMessage = false;
	self.counter = 0;
	self.closeCounter = 0;
	self.currentTrain = 0;
	self.trainLength = 0;
    return true;
end;

function RailRoadCrossing:delete()
	if self.trigger then
		removeTrigger(self.trigger);
	end;
end;

function RailRoadCrossing:readStream(streamId, connection)
end;

function RailRoadCrossing:writeStream(streamId, connection)
end;

function RailRoadCrossing:readUpdateStream(streamId, timestamp, connection)
	RailRoadCrossing:superClass().readUpdateStream(self, streamId, timestamp, connection);
	if connection:getIsServer() then
		self.isClosed = streamReadBool(streamId);
		for i=1, #self.barriers do
			self.barriers[i].closing = streamReadBool(streamId);
		end;
		local showMessage = streamReadBool(streamId);
		if showMessage then
			g_currentMission:showBlinkingWarning(string.format(g_i18n:getText("crossingPenalty"), g_i18n.globalI18N:getCurrencySymbol(true)), 3000);
		end;
	end;
end;

function RailRoadCrossing:writeUpdateStream(streamId, connection, dirtyMask)
	RailRoadCrossing:superClass().writeUpdateStream(self, streamId, connection, dirtyMask);
	if not connection:getIsServer() then
		streamWriteBool(streamId, self.isClosed);
		for i=1, #self.barriers do
			streamWriteBool(streamId, self.barriers[i].closing);
		end;
		streamWriteBool(streamId, self.showMessage);
		if self.showMessage then
			self.showMessage = false;
		end;
	end;
end;

function RailRoadCrossing:update(dt)
	if self.isClosed then
		if self.isServer then
			self.closeCounter = math.min(self.closeCounter + dt, 4000);
			if self.currentTrain > 0 then
				local x, _, z = getWorldTranslation(self.currentTrain);
				local disToCrossing = Utils.vector2Length(x-self.x, z-self.z) - (self.trainLength+50);
				if disToCrossing > 0 and disToCrossing > self.lastDistance then
					self.lastDistance = 1000;
					self.currentTrain = 0;
					self.trainLength = 0;
					self.isClosed = false;
					self:raiseDirtyFlags(self.crossingDirtyFlag);
				else
					self.lastDistance = disToCrossing;
				end;
			end;
			for i=1, #self.barriers do
				local barrier = self.barriers[i];
				if not barrier.isClosed then
					local isClosed = self.closeCounter >= 4000;
					for ii=1, #g_currentMission.vehicles do
						local vehicle = g_currentMission.vehicles[ii];
						local vx, _, vz = getWorldTranslation(vehicle.rootNode);
						local bx, _, bz = worldToLocal(barrier.node, vx, _, vz);
						if bx > -7 and bx < 0 and math.abs(bz) < vehicle.sizeLength then
							isClosed = false;
							break;
						end;
					end;
					if barrier.isClosed ~= isClosed then
						barrier.isClosed = true;
						barrier.closing = true;
						self:raiseDirtyFlags(self.crossingDirtyFlag);
					end;
				end;
			end;
		end;
		if not self.wasClosed then
			self.wasClosed = true;
			if self.ring then setVisibility(self.ring, true); end;
		end;
		self.counter = self.counter + dt;
		if self.counter < 1000 then
			setVisibility(self.lights[1], true);
			setVisibility(self.lights[2], false);
		else
			setVisibility(self.lights[1], false);
			setVisibility(self.lights[2], true);
		end;
		if self.counter > 2000 then
			self.counter = 0;
		end;
	end;
	for i=1, #self.barriers do
		local barrier = self.barriers[i];
		if barrier.opening then
			barrier.position = math.max(barrier.position - barrier.speed * dt, 0);
			setRotation(barrier.node, 0, 0, barrier.position*barrier.endRotZ);
			if barrier.position == 0 then
				barrier.opening = false;
			end;
		end;
		if barrier.closing then
			barrier.position = math.min(barrier.position + barrier.speed * dt, 1);
			setRotation(barrier.node, 0, 0, barrier.position*barrier.endRotZ);
			if barrier.position == 1 then
				barrier.closing = false;
			end;
		end;
	end;
	if self.wasClosed and not self.isClosed then
		setVisibility(self.lights[1], false);
		setVisibility(self.lights[2], false);
		if self.ring then setVisibility(self.ring, false); end;
		for i=1, #self.barriers do
			local barrier = self.barriers[i];
			barrier.opening = true;
			barrier.isClosed = false;
		end;
		self.closeCounter = 0;
		self.wasClosed = false;
	end;
end;

function RailRoadCrossing:triggerCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
	if self.isServer then
		if onEnter then
			local vehicle = g_currentMission.nodeToVehicle[otherShapeId];
			if vehicle and not vehicle.isCpPathvehicle and self.isClosed then
				if self.parent:notCPVehicle(otherShapeId) then
					Economica.statistics.railroadPenalty = Economica.statistics.railroadPenalty + 1;
				end;
				g_currentMission:addSharedMoney(-1000, "other");
				g_currentMission:addMoneyChange(-1000, FSBaseMission.MONEY_TYPE_SINGLE, true, g_i18n:getText("penalties"));
				g_currentMission:showBlinkingWarning(string.format(g_i18n:getText("crossingPenalty"), g_i18n.globalI18N:getCurrencySymbol(true)), 3000);
				self.showMessage = true;
				self:raiseDirtyFlags(self.crossingDirtyFlag);
			end;
		end;
	end;
end;
