--author: igor29381

GaragePlayerTrigger = {}

local GaragePlayerTrigger_mt = Class(GaragePlayerTrigger, Object);

GaragePlayerTrigger.STATE_CLOSED = 1;
GaragePlayerTrigger.STATE_OPENING = 2;
GaragePlayerTrigger.STATE_OPENED = 3;
GaragePlayerTrigger.STATE_CLOSING = 4;

function GaragePlayerTrigger.onCreate(id)
	local trigger = GaragePlayerTrigger:new(g_server ~= nil, g_client ~= nil);
	g_currentMission:addOnCreateLoadedObject(trigger);
	trigger:load(id);
	trigger:register(true);
end;

function GaragePlayerTrigger:new(isServer, isClient, customMt)
	if customMt == nil then
		customMt = GaragePlayerTrigger_mt;
	end;
	local self = Object:new(isServer, isClient, customMt);
	return self;
end;

function GaragePlayerTrigger:load(id)
	self.nodeId = id;
	self.triggerId = getChild(id, "playerTrigger");
	if self.triggerId == 0 then
		self.triggerId = id;
	end;
	addTrigger(self.triggerId, "triggerCallback", self);
	self.gates = {};
	local gatesId = getChild(id, "gates");
	local gatesString = getUserAttribute(gatesId, "gates");
	if gatesString then
		local gates = Utils.splitString(" ", gatesString);
		for i=1, #gates do
			local gate = {};
			gate.node = Utils.indexToObject(gatesId, gates[i]);
			gate.startX, gate.startY, gate.startZ = getRotation(gate.node);
			gate.endX = gate.startX + math.rad(Utils.getNoNil(getUserAttribute(gate.node, "rotX"), "0"));
			gate.endY = gate.startY + math.rad(Utils.getNoNil(getUserAttribute(gate.node, "rotY"), "0"));
			gate.endZ = gate.startZ + math.rad(Utils.getNoNil(getUserAttribute(gate.node, "rotZ"), "0"));
			table.insert(self.gates, gate);
		end;
	else
		for g=1, getNumOfChildren(gatesId) do
			local gate = {};
			gate.node = getChildAt(gatesId, g-1);
			gate.startX, gate.startY, gate.startZ = getRotation(gate.node);
			gate.endX = gate.startX + math.rad(Utils.getNoNil(getUserAttribute(gate.node, "rotX"), "0"));
			gate.endY = gate.startY + math.rad(Utils.getNoNil(getUserAttribute(gate.node, "rotY"), "0"));
			gate.endZ = gate.startZ + math.rad(Utils.getNoNil(getUserAttribute(gate.node, "rotZ"), "0"));
			table.insert(self.gates, gate);
		end;
	end;
	self.startPoint = getChild(id, "startPoint");
	local endPoint = getChild(self.startPoint, "endPoint");
	if endPoint > 0 then
		local ex,_,ez = getTranslation(endPoint);
		self.dimensions = {ex, ez};
	end;
	local appearsOnPDA = Utils.getNoNil(getUserAttribute(id, "appearsOnPDA"), false);
	if appearsOnPDA then
		local x, _, z = getWorldTranslation(id);
		local visitPoint = getChild(id, "visitPoint");
		if visitPoint > 0 then
			x, _, z = getWorldTranslation(visitPoint);
		end;
		local width, height = getNormalizedScreenValues(10, 10);
		local uvs = getNormalizedUVs({135, 940, 46, 46});
		self.mapHotspot = g_currentMission.ingameMap:createMapHotspot("", "", g_baseUIFilename, uvs, nil, x, z, width, height, false, false, true, nil, true);
	end;
	self.sound = getChild(id, "sound");
	if self.sound > 0 then setVisibility(self.sound, false); end;
	self.light = getChild(id, "light");
	if self.light > 0 then setVisibility(self.light, false); end;
	self.vehiclesString = "";
	self.state = GaragePlayerTrigger.STATE_CLOSED;
	self.gatePos = 0;
	self.speed = Utils.getNoNil(getUserAttribute(id, "speed"), 0.0003);
	self.storeNums = {};
	self.storeTexts = {};
	self.storeImages = {};
	self.player = false;
	self.isEnabled = true;
	return true;
end;

function GaragePlayerTrigger:delete()
	removeTrigger(self.triggerId);
	GaragePlayerTrigger:superClass().delete(self);
end;

function GaragePlayerTrigger:readStream(streamId, connection)
	GaragePlayerTrigger:superClass().readStream(self, streamId);
	if connection:getIsServer() then
		self.state = streamReadUIntN(streamId, 4);
		local count = streamReadInt8(streamId);
		if count > 0 then
			for i=1, count do
				local storeNum = streamReadInt32(streamId);
				self:insertStoreItems(storeNum);
			end;
		end;
		if count == 0 then
			UniversalFactoryHUD.garageOverlays = nil;
			UniversalFactoryHUD.garageTexts = nil;
		end;
	end;
end;

function GaragePlayerTrigger:writeStream(streamId, connection)
	GaragePlayerTrigger:superClass().writeStream(self, streamId);
	if not connection:getIsServer() then
		local state = GaragePlayerTrigger.STATE_CLOSED;
		if self.state == GaragePlayerTrigger.STATE_OPENED or self.state == GaragePlayerTrigger.STATE_OPENING then
			state = GaragePlayerTrigger.STATE_OPENING;
		end;
		streamWriteUIntN(streamId, state, 4);
		local count = #self.storeNums;
		streamWriteInt8(streamId, count);
		if count > 0 then
			for i=1, count do
				streamWriteInt32(streamId, self.storeNums[i]);
			end;
		end;
	end;
end;

function GaragePlayerTrigger:update(dt)
	if self.player then
		setTextColor(1, 1, 1, 1);
		setTextAlignment(RenderText.ALIGN_CENTER);
		if self.state == GaragePlayerTrigger.STATE_OPENED or self.state == GaragePlayerTrigger.STATE_OPENING then
			renderText(0.5, 0.28, 0.02, g_i18n:getText("shutGates"));
		elseif self.state == GaragePlayerTrigger.STATE_CLOSED or self.state == GaragePlayerTrigger.STATE_CLOSING then
			renderText(0.5, 0.28, 0.02, g_i18n:getText("openGates"));
		end;
		if #self.storeTexts > 0 then
			UniversalFactoryHUD.garageOverlays = self.storeImages;
			UniversalFactoryHUD.garageTexts = self.storeTexts;
		else
			renderText(0.5, 0.3, 0.02, g_i18n:getText("empty"));
		end;
		setTextAlignment(RenderText.ALIGN_LEFT);
		if InputBinding.hasEvent(InputBinding.TOGGLE_PIPE) then
			if self.isServer then
				self:onChangeState();
			else
				g_client:getServerConnection():sendEvent(GaragePlayerEvent:new(true, 0, 0, self));
			end;
		end;
	end;
	if self.state == GaragePlayerTrigger.STATE_OPENING then
		self.gatePos = math.min(self.gatePos + self.speed * dt, 1);
		if self.gatePos == 1 then
			self.state = GaragePlayerTrigger.STATE_OPENED;
			if self.sound > 0 then setVisibility(self.sound, false); end;
			if self.light > 0 then setVisibility(self.light, true); end;
		end;
	end;
	if self.state == GaragePlayerTrigger.STATE_CLOSING then
		self.gatePos = math.max(self.gatePos - self.speed * dt, 0);
		if self.gatePos == 0 then
			self.state = GaragePlayerTrigger.STATE_CLOSED;
			if self.sound > 0 then setVisibility(self.sound, false); end;
			if self.isServer then self:onEndClosing(true); end;
		end;
	end;
	if self.gatePos > 0 and self.gatePos < 1 then
		for _, gate in pairs(self.gates) do
			local x = (1 - self.gatePos) * gate.startX + self.gatePos * gate.endX;
			local y = (1 - self.gatePos) * gate.startY + self.gatePos * gate.endY;
			local z = (1 - self.gatePos) * gate.startZ + self.gatePos * gate.endZ;
			setRotation(gate.node, x, y, z);
		end;
		if self.sound > 0 then setVisibility(self.sound, true); end;
	end;
end;

function GaragePlayerTrigger:onChangeState()
	if self.state == GaragePlayerTrigger.STATE_CLOSED then
		self.state = GaragePlayerTrigger.STATE_OPENING;
		self:onStartOpening();
	elseif self.state == GaragePlayerTrigger.STATE_OPENED then
		if self.light > 0 then setVisibility(self.light, false); end;
		self.state = GaragePlayerTrigger.STATE_CLOSING;
		g_server:broadcastEvent(GaragePlayerEvent:new(false, GaragePlayerTrigger.STATE_CLOSING, 0, self));
	end;
end;

function GaragePlayerTrigger:onStartOpening()
	if self.vehiclesString ~= "" then
		local folderName = g_currentMission.missionInfo.savegameDirectory;
		if not folderName then
			folderName = string.format("%s/savegame%d", getUserProfileAppPath(), g_currentMission.missionInfo.savegameIndex);
			createFolder(folderName);
		end;
		local fileName = string.format("%s/VehicleInGarage.xml", folderName);
		local file = io.open(fileName, "w");
		file:write('<?xml version="1.0" encoding="utf-8" standalone="no" ?>\n<vehicles>\n', self.vehiclesString, '\n</vehicles>');
		file:close();
		local xmlFile = loadXMLFile("tempXML", fileName);
		local i = 0;
		while true do
			local key = string.format("vehicles.vehicle(%d)", i);
			if not hasXMLProperty(xmlFile, key) then break; end;
			local vehicleXML = Utils.convertFromNetworkFilename(getXMLString(xmlFile, key.."#filename"));
			if vehicleXML then
				local x,_,z = getWorldTranslation(self.nodeId);
				g_currentMission:loadVehicle(vehicleXML, x, 0.2, z, 0, true, 0, 0, nil, {xmlFile=xmlFile, key=key, resetVehicles=false});
			end;
			i=i+1;
		end;
		delete(xmlFile);
		self.vehiclesString = "";
		UniversalFactoryHUD.garageOverlays = nil;
		UniversalFactoryHUD.garageTexts = nil;
		self.storeNums = {};
		self.storeTexts = {};
		self.storeImages = {};
	end;
	g_server:broadcastEvent(GaragePlayerEvent:new(false, GaragePlayerTrigger.STATE_OPENING, 0, self));
end;

function GaragePlayerTrigger:onEndClosing(sendEvent)
	local vehicles = {};
	for i=1, #g_currentMission.vehicles do
		local vehicle = g_currentMission.vehicles[i];
		if self.dimensions and not (vehicle.leasingInfos and vehicle.leasingInfos.leased) then
			local vx1, _, vz1 = worldToLocal(self.startPoint, getWorldTranslation(vehicle.rootNode));
			if vx1 > 0 and vz1 > 0 and vx1 < self.dimensions[1] and vz1 < self.dimensions[2] then
				table.insert(vehicles, vehicle);
			end;
		end;
	end;
	if #vehicles > 0 then
		local vehiclesString = "";
		for i=1, #vehicles do
			local vehicle = vehicles[i];
			local attributes, nodes = vehicle:getSaveAttributesAndNodes("       ");
			attributes = string.format('<vehicle filename="%s" %s>%s\n</vehicle>', Utils.encodeToHTML(Utils.convertToNetworkFilename(vehicle.configFileName)), attributes, nodes);
			if vehiclesString == "" then
				vehiclesString = attributes;
			else
				vehiclesString = string.format("%s\n%s", vehiclesString, attributes);
			end;
			self.vehiclesString = vehiclesString;
			local storeNum = StoreItemsUtil.storeItemsByXMLFilename[string.lower(vehicle.configFileName)].id;
			self:insertStoreItems(storeNum);
			g_currentMission:removeVehicle(vehicle);
			if sendEvent then
				g_server:broadcastEvent(GaragePlayerEvent:new(false, 0, storeNum, self));
			end;
		end;
	end;
end;

function GaragePlayerTrigger:insertStoreItems(storeNum)
	local storeItem = StoreItemsUtil.storeItems[storeNum];
	local storeText = "Unknown vehicle";
	local storeImage = Garage.main.unknownVehicleImage;
	if storeItem then
		storeText = string.format("%s %s", storeItem.brand, storeItem.name);
		if storeItem.imageActive ~= "" then
			storeImageActive = createImageOverlay(Utils.getFilename(storeItem.imageActive));
			if storeImageActive and storeImageActive ~= 0 then
				storeImage = storeImageActive;
			end;
		end;
	end;
	table.insert(self.storeNums, storeNum);
	table.insert(self.storeTexts, storeText);
	table.insert(self.storeImages, storeImage);
end;

function GaragePlayerTrigger:triggerCallback(triggerId, otherActorId, onEnter, onLeave, onStay, otherShapeId)
	if self.garage.isOwned then
		if onEnter then
			if g_currentMission.player.rootNode == otherShapeId then
				self.player = true;
			end;
		elseif onLeave then
			self.player = false;
			UniversalFactoryHUD.garageOverlays = nil;
			UniversalFactoryHUD.garageTexts = nil;
		end;
	end;
end;
-------------------------------------------------------------------------------------------------
GaragePlayerEvent = {};
GaragePlayerEvent_mt = Class(GaragePlayerEvent, Event);

InitEventClass(GaragePlayerEvent, "GaragePlayerEvent");

function GaragePlayerEvent:emptyNew()
	local self = Event:new(GaragePlayerEvent_mt);
    return self;
end;

function GaragePlayerEvent:new(changeState, state, storeNum, trigger)
	local self = GaragePlayerEvent:emptyNew();
	self.changeState = changeState;
	self.state = state;
	self.storeNum = storeNum;
	self.trigger = trigger;
	return self;
end;

function GaragePlayerEvent:readStream(streamId, connection)
	local changeState = streamReadBool(streamId);
	local state = streamReadInt8(streamId);
	local storeNum = streamReadInt32(streamId);
	local id = streamReadInt32(streamId);
    local trigger = networkGetObject(id);
	if trigger ~= nil then
		if g_server then
			if changeState then
				trigger:onChangeState();
			end;
		else
			if storeNum > 0 then
				trigger:insertStoreItems(storeNum);
			else
				trigger.storeNums = {};
				trigger.storeTexts = {};
				trigger.storeImages = {};
				UniversalFactoryHUD.garageOverlays = nil;
				UniversalFactoryHUD.garageTexts = nil;
			end;
		end;
		if state > 0 then
			trigger.state = state;
			if state == GaragePlayerTrigger.STATE_CLOSING then
				if trigger.light > 0 then setVisibility(trigger.light, false); end;
			end;
		end;
	end;
end;

function GaragePlayerEvent:writeStream(streamId, connection)
	streamWriteBool(streamId, self.changeState);
	streamWriteInt8(streamId, self.state);
	streamWriteInt32(streamId, self.storeNum);
	streamWriteInt32(streamId, networkGetObjectId(self.trigger));
end;
