
function AnimalHusbandry:load(typeName, transformId, xmlFile, customEnvironment, baseDirectory)
    self.nodeId = transformId;
	if g_currentMission.husbandries[typeName] ~= nil then
        print("Error: duplicate animal husbandry of type "..typeName);
        return false;
    end
    if AnimalUtil.animals[typeName] == nil then
        print("Error: animal type '"..typeName.."' not found");
    end
    self.baseDirectory = baseDirectory;
    self.customEnvironment = customEnvironment;
    local navMesh = Utils.indexToObject(transformId, getUserAttribute(transformId, "navMeshIndex"));
    if navMesh == nil then
        print("Error: invalid navMeshIndex in "..getName(transformId));
        return false;
    end
    self.animalDesc = AnimalUtil.animals[typeName];
    self.typeName = typeName;
    g_currentMission.husbandries[typeName] = self;
    local placementRaycastDistance = getUserAttribute(transformId, "placementRaycastDistance");
    if placementRaycastDistance == nil then
        placementRaycastDistance = 2;
    end
    if navMesh ~= nil then
        self.husbandryId = createAnimalHusbandry(typeName, navMesh, xmlFile, placementRaycastDistance, not g_currentMission.missionDynamicInfo.isMultiplayer);
    end
    self.numSubTypes = 2;
    self.numAnimals = {};
    self.numVisibleAnimals = {}
    for i=1,self.numSubTypes do
        self.numAnimals[i-1] = 0;
        self.numVisibleAnimals[i-1] = 0;
    end
    self.totalNumAnimals = 0;
    self.tipTriggers = {};
    local tipTriggers = Utils.indexToObject(transformId, getUserAttribute(transformId, "tipTriggersIndex"));
    if tipTriggers ~= nil then
        for i=1, getNumOfChildren(tipTriggers) do
            local tipTriggerId = getChildAt(tipTriggers, i-1);
            local tipTrigger = FeedingTroughTipTrigger:new(self.isServer, self.isClient);
            tipTrigger:load(tipTriggerId, self);
            g_currentMission:addOnCreateLoadedObject(tipTrigger);
            tipTrigger:register(true);
            if next(tipTrigger.acceptedFillTypes) == nil then
                tipTrigger:addFoodGroupsAndMixtures(FillUtil.foodGroups[self.animalDesc.index], FillUtil.foodMixtures[self.animalDesc.index]);
            end
            table.insert(self.tipTriggers, tipTrigger);
        end
    end
    self.dailyUpkeep = self.animalDesc.dailyUpkeep;
    self.automaticallySellMilk = false;
    self.cleanlinessFactor = 1.0;
    self.newAnimalPercentage = 0.0;
    self.fillTypes = {};
    if self.animalDesc.waterPerDay > 0 then
        table.insert(self.fillTypes, FillUtil.FILLTYPE_WATER);
    end;
    if self.animalDesc.strawPerDay > 0 then
        table.insert(self.fillTypes, FillUtil.FILLTYPE_STRAW);
    end;
    if self.animalDesc.foodPerDay > 0 and FillUtil.foodGroups[self.animalDesc.index] ~= nil then
        for _,foodGroup in pairs(FillUtil.foodGroups[self.animalDesc.index]) do
            for fillType,state in pairs(foodGroup.fillTypes) do
                if state == true then
                    local fillTypePresent = false;
                    for _,fType in pairs(self.fillTypes) do
                        if fType == fillType then
                            fillTypePresent = true;
                            break;
                        end;
                    end;
                    if not fillTypePresent then
                        table.insert(self.fillTypes, fillType);
                    end;
                end;
            end;
        end;
    end;
    local manureAreaNode = Utils.indexToObject(transformId, getUserAttribute(transformId, "manureHeapArea"));
    if manureAreaNode ~= nil then
        local start = getChildAt(manureAreaNode, 0);
        local width = getChildAt(manureAreaNode, 1);
        local height = getChildAt(manureAreaNode, 2);
        if start ~= nil and width ~= nil and height ~= nil then
            self.manureArea = {start=start, width=width, height=height};
            self.manureFillLevel = 0;                                                                               -- manure in manureHeapArea
            g_currentMission:addManureHeap("$l10n_ui_"..self.typeName.."ManureHeap", self);
            local fillTypes = {};
            fillTypes[FillUtil.FILLTYPE_MANURE] = true;
            TipUtil.setFixedFillTypesArea(self.manureArea, fillTypes)
        end;
    end;
    self.palletSpawnerId = Utils.indexToObject(transformId, getUserAttribute(transformId, "palletSpawnerIndex"));
    if self.palletSpawnerId ~= nil then
        self.palletFillType = FillUtil.FILLTYPE_WOOL;
        local fillTypeStr = getUserAttribute(self.palletSpawnerId, "palletFillType");
        if fillTypeStr ~= nil then
            local fillTypeInt = FillUtil.fillTypeNameToInt[fillTypeStr];
            if fillTypeInt ~= nil then
                self.palletFillType = fillTypeInt;
            end;
        end;
        if self.isServer then
            self.palletSpawnerNode = getChildAt(self.palletSpawnerId, 0);
            self.palletSpawnerAreaSizeX = Utils.getNoNil(getUserAttribute(self.palletSpawnerId, "spawnerAreaSizeX"), 5);
            self.palletSpawnerAreaSizeZ = Utils.getNoNil(getUserAttribute(self.palletSpawnerId, "spawnerAreaSizeZ"), 5);
            self.palletSpawnerFillDelta = 0;
        else
            self.currentPalletFillPercentage = 0;
        end;
    end;
    self.numObjectsInPalletSpawnerTrigger = 0;
    self.pickupObjectsId = Utils.indexToObject(transformId, getUserAttribute(transformId, "pickupObjectsIndex"));
    if self.pickupObjectsId ~= nil then
        if self.typeName == "chicken" then
            if g_addTestCommands then
                addConsoleCommand("gsSpawnPickupObjects", "Spawn pickup objects", "consoleCommandSpawnPickupObjects", self);
            end
        end
        self.pickupObjectsFillType = FillUtil.FILLTYPE_EGG;
        local fillTypeStr = getUserAttribute(self.pickupObjectsId, "pickupObjectsFillType");
        if fillTypeStr ~= nil then
            local fillTypeInt = FillUtil.fillTypeNameToInt[fillTypeStr];
            if fillTypeInt ~= nil then
                self.pickupObjectsFillType = fillTypeInt;
            end
        end
        local numPickupObjects = getNumOfChildren(self.pickupObjectsId);
        if self.isServer then
            self.pickupObjectsToActivate = {};
            self.numPickupObjectsToSpawn = 0;
        end
        self.numActivePickupObjects = 0;
        for i=1, numPickupObjects do
            local pickupObject = getChildAt(self.pickupObjectsId, i-1);
            setVisibility(pickupObject, false);
            if self.isServer then
                table.insert(self.pickupObjectsToActivate, pickupObject);
                local triggerId = getChildAt(pickupObject, 0);
                addTrigger(triggerId, "pickupObjectTriggerCallback", self);
            end
        end
    end
    self.dirtAreas = {};
    self.numDirtAreas = 0;
    self.dirtToDrop = 0;
    local dirtAreas = Utils.indexToObject(transformId, getUserAttribute(transformId, "dirtAreas"));
    if dirtAreas ~= nil then
        for i=0,getNumOfChildren(dirtAreas)-1 do
            local node = getChildAt(dirtAreas, i);
            local start = getChildAt(node, 0);
            local width = getChildAt(node, 1);
            local height = getChildAt(node, 2);
            if start ~= nil and width ~= nil and height ~= nil then
                table.insert(self.dirtAreas, {start=start, width=width, height=height});
                self.numDirtAreas = self.numDirtAreas + 1;
            end
        end
    end
    local dirtificationFillType = getUserAttribute(transformId, "dirtificationFillType");
    if dirtificationFillType ~= nil then
        if FillUtil.fillTypeNameToInt[dirtificationFillType] ~= nil then
            self.dirtificationFillType = FillUtil.fillTypeNameToInt[dirtificationFillType];
        end
    end
    if self.dirtificationFillType == nil then
        if self.typeName == "pig" then
            self.dirtificationFillType = FillUtil.FILLTYPE_MAIZE;
        else
            self.dirtificationFillType = FillUtil.FILLTYPE_GRASS_WINDROW;
        end
    end
    --local fillTypes = {};
    --fillTypes[self.dirtificationFillType] = true;
    --for _,dirtArea in pairs(self.dirtAreas) do
    --    TipUtil.setFixedFillTypesArea(dirtArea, fillTypes)
    --end
    self.tipTriggersFillLevels = {};
    for _,fillType in pairs(self.fillTypes) do
        self.tipTriggersFillLevels[fillType] = {};
    end
    for fillType, fillLevels in pairs(self.tipTriggersFillLevels) do
        for i,tipTrigger in ipairs(self.tipTriggers) do
            if tipTrigger.acceptedFillTypes[fillType] then
                table.insert(fillLevels, {tipTrigger=tipTrigger, tipTriggerIndex=i, fillLevel=0});
            end;
        end;
    end;
    self.strawPlaneId = Utils.indexToObject(transformId, getUserAttribute(transformId, "strawPlaneIndex"));
    if self.strawPlaneId ~= nil then
        local minY, maxY = Utils.getVectorFromString(getUserAttribute(transformId, "strawPlaneMinMaxY"));
        self.strawPlaneMinY = Utils.getNoNil(minY, 0);
        self.strawPlaneMaxY = Utils.getNoNil(maxY, self.strawPlaneMinY+0.1);
        self.strawPlaneMaxFillLevel = Utils.getNoNil(getUserAttribute(transformId, "strawPlaneMaxFillLevel"), 18000);
        self:updateStrawPlane();
    end
    local numAnimals = getUserAttribute(transformId, "initialNumAnimals");
    if numAnimals ~= nil and numAnimals > 0 then
        self:addAnimals(numAnimals, 0);
    end
    --
    self.areasUpdateTimer = 2000;
    self.dirtToDrop = 0;
    self.manureToDrop = 0;
    self.manureToRemove = 0
    self.updateMinutesInterval = 20;
    self.updateMinutes = 0;
    self.reproductionRatePerDay = 0;
    self.productivity = 0;
    if not g_isPresentationVersion then
        if self.typeName == "chicken" then
            self:addAnimals(1, 1);
            self:addAnimals(24, 0);
        end
    end
    self.hasStatistics = self.animalDesc.hasStatistics
    if self.isServer then
        g_currentMission.environment:addMinuteChangeListener(self);
    end
    local numHusbandries = 0;
    for _ in pairs(g_currentMission.husbandries) do
        numHusbandries = numHusbandries + 1;
    end
    self.saveId = Utils.getNoNil(getUserAttribute(transformId, "saveId"), "Animals_"..self.typeName);
    return true;
end;

function AnimalHusbandry:loadFromAttributesAndNodes(xmlFile, key)
    for i=1, self.numSubTypes do
        local numAnimals = getXMLInt(xmlFile, key..string.format('#numAnimals%d', i-1));
        if numAnimals ~= nil and numAnimals >= 0 then
            local diff = numAnimals - self.numAnimals[i-1];
            if diff > 0 then
                self:addAnimals(diff, i-1);
            elseif diff < 0 then
                self:removeAnimals(-diff, i-1);
            end
        end
    end
    self.newAnimalPercentage = Utils.getNoNil(getXMLFloat(xmlFile, key.."#newAnimalPercentage"), 0);
    self.reproductionRatePerDay = 0;
    self.cleanlinessFactor = Utils.getNoNil(getXMLFloat(xmlFile, key.."#cleanlinessFactor"), self.cleanlinessFactor);
    local dirtToDrop = getXMLFloat(xmlFile, key.."#dirtToDrop");
    if dirtToDrop ~= nil then
        self.dirtToDrop = dirtToDrop;
    end
    if self.pickupObjectsId ~= nil then
        local numActivePickupObjects = getXMLInt(xmlFile, key.."#numActivePickupObjects");
        if numActivePickupObjects ~= nil then
            self:spawnPickupObjects(numActivePickupObjects);
        end
    end
    local newFillLevelsLoaded = false;
    local i = 0;
    while true do
        local fillLevelKey = key..string.format(".tipTriggerFillLevel(%d)", i);
        if not hasXMLProperty(xmlFile, fillLevelKey) then
            break
        end
        local fillTypeName = getXMLString(xmlFile, fillLevelKey.."#fillType");
        local fillLevel = getXMLFloat(xmlFile, fillLevelKey.."#fillLevel");
        local tipTriggerIndex = getXMLInt(xmlFile, fillLevelKey.."#tipTriggerIndex");
        if fillTypeName ~= nil and fillLevel ~= nil and tipTriggerIndex ~= nil then
            newFillLevelsLoaded = true;
            local fillType = FillUtil.fillTypeNameToInt[fillTypeName];
            if fillType ~= nil then
                local fillLevels = self.tipTriggersFillLevels[fillType];
                if fillLevels ~= nil then
                    for _,data in pairs(fillLevels) do
                        if data.tipTriggerIndex == tipTriggerIndex then
                            data.fillLevel = fillLevel;
                            break;
                        end
                    end
                end
            end
        end
        i = i + 1;
    end
    if not newFillLevelsLoaded then
        local i = 0;
        while true do
            local fillLevelKey = key..string.format(".fillLevel(%d)", i);
            if not hasXMLProperty(xmlFile, fillLevelKey) then
                break
            end
            local fillTypeName = getXMLString(xmlFile, fillLevelKey.."#fillType");
            local fillLevel = getXMLFloat(xmlFile, fillLevelKey.."#fillLevel");
            if fillTypeName ~= nil and fillLevel ~= nil then
                local fillType = FillUtil.fillTypeNameToInt[fillTypeName];
                if fillType ~= nil then
                    local fillLevels = self.tipTriggersFillLevels[fillType];
                    if fillLevels ~= nil then
                        local numTriggers = table.getn(fillLevels);
                        if numTriggers > 0 then
                            for _,data in pairs(fillLevels) do
                                data.fillLevel = fillLevel/numTriggers;
                            end
                        end
                    end
                end
            end
            i = i + 1;
        end
    end
    self:updateStrawPlane();
    for _,trigger in pairs(self.tipTriggers) do
        trigger:updateFillPlane();
    end;
	if self.animalDesc.manurePerDay > 0 and self.parent.fillType[FillUtil.FILLTYPE_MANURE] then
        local manureToDrop = getXMLFloat(xmlFile, key.."#manureToDrop");
        if manureToDrop then
            self.manureToDrop = manureToDrop;
        end
		self:updateManureStatistics();
	end;
    return true;
end;

function AnimalHusbandry:getSaveAttributesAndNodes(nodeIdent)
    local attributes = '';
    local nodes = "";
    for i=1, self.numSubTypes do
        attributes = attributes .. string.format(' numAnimals%d="%d"', i-1, self.numAnimals[i-1]);
    end
    if self.newAnimalPercentage ~= nil then
        attributes = attributes .. ' newAnimalPercentage="'..self.newAnimalPercentage..'"';
    end
    if self.cleanlinessFactor ~= nil then
        attributes = attributes .. ' cleanlinessFactor="'..self.cleanlinessFactor..'"';
    end
    if self.dirtToDrop ~= nil then
        attributes = attributes .. ' dirtToDrop="' .. self.dirtToDrop .. '"';
    end;
	if self.animalDesc.manurePerDay > 0 and self.parent.fillType[FillUtil.FILLTYPE_MANURE] then
		attributes = attributes .. ' manureToDrop="' .. self.manureToDrop ..'"';
	end;
    if self.pickupObjectsId ~= nil then
        attributes = attributes..' numActivePickupObjects="'..self.numActivePickupObjects..'"';
    end
    for fillType, fillLevels in pairs(self.tipTriggersFillLevels) do
        local fillTypeName = FillUtil.fillTypeIntToName[fillType];
        if fillTypeName ~= nil then
            for _, data in ipairs(fillLevels) do
                if nodes:len() > 0 then
                    nodes = nodes.."\n";
                end
                nodes = nodes.. nodeIdent..'<tipTriggerFillLevel fillType="'..fillTypeName..'" tipTriggerIndex="'..data.tipTriggerIndex..'" fillLevel="'..data.fillLevel..'" />';
            end
        end
    end
    return attributes, nodes;
end;

function AnimalHusbandry:readStream(streamId, connection)
    AnimalHusbandry:superClass().readStream(self, streamId, connection);
    if connection:getIsServer() then
        for i=1, self.numSubTypes do
            local numAnimals = streamReadUInt16(streamId);
            local diff = numAnimals - self.numAnimals[i-1];
            if diff > 0 then
                self:addAnimals(diff, i-1);
            elseif diff < 0 then
                self:removeAnimals(-diff, i-1);
            end
        end
        -- read fill levels
        if self.cleanlinessFactor ~= nil then
            self.cleanlinessFactor = streamReadFloat32(streamId);
        end
        for _,fillType in pairs(self.fillTypes) do
            self:streamReadTipTriggersFillLevels(streamId, fillType);
        end
        if self.animalDesc.strawPerDay > 0 then
            self:updateStrawPlane();
        end
        self.reproductionRatePerDay = streamReadFloat32(streamId);
        self.newAnimalPercentage = streamReadFloat32(streamId);
        if self.pickupObjectsId ~= nil then
            local numPickupObjects = getNumOfChildren(self.pickupObjectsId);
            for i=1, numPickupObjects do
                local pickupObject = getChildAt(self.pickupObjectsId, i-1);
                setVisibility(pickupObject, streamReadBool(streamId));
            end
            self.numActivePickupObjects = streamReadUInt16(streamId);
        end
        for _,trigger in pairs(self.tipTriggers) do
            trigger:updateFillPlane();
        end
    end
end;

function AnimalHusbandry:writeStream(streamId, connection)
    AnimalHusbandry:superClass().writeStream(self, streamId, connection);
    if not connection:getIsServer() then
        for i=1, self.numSubTypes do
            streamWriteUInt16(streamId, self.numAnimals[i-1]);
        end;
        -- write fill levels
        if self.cleanlinessFactor ~= nil then
            streamWriteFloat32(streamId, self.cleanlinessFactor);
        end;
        for _,fillType in pairs(self.fillTypes) do
            self:streamWriteTipTriggersFillLevels(streamId, fillType);
        end;
        streamWriteFloat32(streamId, self.reproductionRatePerDay);
        streamWriteFloat32(streamId, self.newAnimalPercentage);
        if self.pickupObjectsId ~= nil then
            local numPickupObjects = getNumOfChildren(self.pickupObjectsId);
            for i=1, numPickupObjects do
                local pickupObject = getChildAt(self.pickupObjectsId, i-1);
                streamWriteBool(streamId, getVisibility(pickupObject));
            end;
            streamWriteUInt16(streamId, self.numActivePickupObjects);
        end;
    end;
end;

function AnimalHusbandry:readUpdateStream(streamId, timestamp, connection)
    AnimalHusbandry:superClass().readUpdateStream(self, streamId, timestamp, connection);
    if connection:getIsServer() then
        if streamReadBool(streamId) then
            -- read fill levels
            if self.cleanlinessFactor ~= nil then
                self.cleanlinessFactor = streamReadFloat32(streamId);
            end;
            for _,fillType in pairs(self.fillTypes) do
                self:streamReadTipTriggersFillLevels(streamId, fillType);
            end;
            if self.animalDesc.strawPerDay > 0 then
                self:updateStrawPlane();
            end;
            self.reproductionRatePerDay = streamReadFloat32(streamId);
            self.newAnimalPercentage = streamReadFloat32(streamId);
            if self.parent.fillType[FillUtil.FILLTYPE_WOOL] then
                self.currentPalletFillPercentage = streamReadUInt8(streamId);
            end;
            if self.typeName ~= "chicken" then
                self.productivity = streamReadUInt8(streamId)/100;
            end;
            if self.pickupObjectsId ~= nil then
                self.numActivePickupObjects = streamReadUInt16(streamId);
            end;
            for _,trigger in pairs(self.tipTriggers) do
                trigger:updateFillPlane();
            end;
        end
        if streamReadBool(streamId) then
            for i=1, self.numSubTypes do
                local numAnimals = streamReadUInt16(streamId);
                local diff = numAnimals - self.numAnimals[i-1];
                if diff > 0 then
                    self:addAnimals(diff, i-1);
                elseif diff < 0 then
                    self:removeAnimals(-diff, i-1);
                end;
            end;
        end;
    end;
end;

function AnimalHusbandry:writeUpdateStream(streamId, connection, dirtyMask)
    AnimalHusbandry:superClass().writeUpdateStream(self, streamId, connection, dirtyMask);
    if not connection:getIsServer() then
        if streamWriteBool(streamId, bitAND(dirtyMask, self.husbandryDirtyFlag) ~= 0) then
            -- write fill levels
            if self.cleanlinessFactor ~= nil then
                streamWriteFloat32(streamId, self.cleanlinessFactor);
            end;
            for _,fillType in pairs(self.fillTypes) do
                self:streamWriteTipTriggersFillLevels(streamId, fillType);
            end;
            streamWriteFloat32(streamId, self.reproductionRatePerDay);
            streamWriteFloat32(streamId, self.newAnimalPercentage);
            if self.parent.fillType[FillUtil.FILLTYPE_WOOL] then
				local fillPercentage = math.floor(100*self.parent.fillType[FillUtil.FILLTYPE_WOOL].level / self.parent.fillType[FillUtil.FILLTYPE_WOOL].capacity);
                streamWriteUInt8(streamId, fillPercentage);
            end;
            if self.typeName ~= "chicken" then
                streamWriteUInt8(streamId, math.floor(self.productivity*100));
            end;
            if self.pickupObjectsId ~= nil then
                streamWriteUInt16(streamId, self.numActivePickupObjects);
            end;
        end;
        if streamWriteBool(streamId, bitAND(dirtyMask, self.husbandryAnimalCountDirtyFlag) ~= 0) then
            for i=1, self.numSubTypes do
                streamWriteUInt16(streamId, self.numAnimals[i-1]);
            end;
        end;
    end;
end;

function AnimalHusbandry:updateTick(dt)
    if self.dirtToDrop > TipUtil.getMinValidLiterValue(self.dirtificationFillType) then
        local dropped = self:updateDirt(math.min(self.dirtToDrop, 20*TipUtil.getMinValidLiterValue(self.dirtificationFillType)));
        self.dirtToDrop = self.dirtToDrop - dropped;
    end;
    if self.manureToDrop > TipUtil.getMinValidLiterValue(FillUtil.FILLTYPE_MANURE) then
        local dropped = self:updateManure(math.min(self.manureToDrop, 20*TipUtil.getMinValidLiterValue(self.dirtificationFillType)));
        self.manureToDrop = self.manureToDrop - dropped;
		self:updateManureStatistics();
    end;
end;

function AnimalHusbandry:updateManureStatistics()
	self.parent.fillType[FillUtil.FILLTYPE_MANURE].level = self:getManureLevel() + self.manureToDrop;
end;

function AnimalHusbandry:minuteChanged()
    if not self.isServer then
        return;
    end
    self.updateMinutes = self.updateMinutes + 1;
    if self.updateMinutes >= self.updateMinutesInterval then
        self.updateMinutes = 0;
        self.productivity = 0;
        local numProducingAnimals = self.totalNumAnimals;
        if self.typeName == "chicken" then
            -- roosters do not procuce stuff eggs
            numProducingAnimals = numProducingAnimals - self.numAnimals[1];
        end
        if self.totalNumAnimals == 0 then
            self.reproductionRatePerDay = 0;
        else
            local dayToInterval = self.updateMinutesInterval/(24*60);
            local strawMultiplier = 1;
            if self.animalDesc.strawPerDay > 0 then
                local straw = self:getFillLevel(FillUtil.FILLTYPE_STRAW);
                local strawNeeded = self.totalNumAnimals * self.animalDesc.strawPerDay * dayToInterval;
                strawMultiplier = math.min(straw / strawNeeded, 1);
                local strawUsage = math.min(strawNeeded, straw);
                if strawUsage > 0 then
                    self:changeFillLevels(-strawUsage, FillUtil.FILLTYPE_STRAW);
                    self:updateStrawPlane();
                end;
            end
            if self.animalDesc.foodPerDay > 0 then
                local foodHappyness = 0.0;
                local totalAmountNeeded = self.animalDesc.foodPerDay * self.totalNumAnimals * dayToInterval;
                for _,foodGroup in pairs(FillUtil.foodGroups[self.animalDesc.index]) do
                    local amountNeeded = totalAmountNeeded * foodGroup.weight;
                    local factor = self:consumeFillTypes(foodGroup.fillTypes, amountNeeded);
                    foodHappyness = foodHappyness + factor * foodGroup.weight;
                end
                self.productivity = 0.9 * foodHappyness + 0.1 * strawMultiplier;
            else
                self.productivity = 1.0;
            end
            if self.animalDesc.dirtFillLevelPerDay > 0 and self.numDirtAreas > 0 then
                -- animals need a clean feeding place
                local dirtIncrease = self.totalNumAnimals * self.animalDesc.dirtFillLevelPerDay * dayToInterval;
                local maxDirtLevel = self.totalNumAnimals * self.animalDesc.dirtFillLevelPerDay;
                local available = 0;
                if self.dirtificationFillType ~= FillUtil.FILLTYPE_MANURE then
                    for _,foodGroup in pairs(FillUtil.foodGroups[self.animalDesc.index]) do
                        for fillType,_ in pairs(foodGroup.fillTypes) do
                            available = available + self:getFillLevel(fillType);
                        end
                    end
                else
                    available = self:getFillLevel(FillUtil.FILLTYPE_STRAW);
                end
                dirtIncrease = math.min(dirtIncrease, available);
                local dirtLevel = self:getDirtLevel();
                self.cleanlinessFactor = 1.0 - math.min(1.0, (dirtLevel/maxDirtLevel));
                if self.cleanlinessFactor < 0.1 then
                    self.productivity = self.productivity * 0.9;
                end
                -- consume
                local fillType = self.dirtificationFillType
                local consumed = dirtIncrease;
                if fillType ~= FillUtil.FILLTYPE_MANURE then
                    local oldFillLevel = self:getFillLevel(fillType);
                    self:changeFillLevels(-math.min(oldFillLevel, dirtIncrease), fillType);
                    local newFillLevel = self:getFillLevel(fillType);
                    consumed = (oldFillLevel - newFillLevel);
                    local remaining = dirtIncrease - consumed;
                    while remaining > 1 do
                        local sumP = 0;
                        for _,foodGroup in pairs(FillUtil.foodGroups[self.animalDesc.index]) do
                            local delta = remaining;
                            local p = self:consumeFillTypes(foodGroup.fillTypes, delta);
                            sumP = sumP + p;
                            consumed = consumed + (p*delta);
                            remaining = remaining - (p*delta);
                            if remaining < 1 then
                                break;
                            end
                        end
                        if remaining < 1 or sumP == 0 then
                            break;
                        end
                    end
                    for _,trigger in pairs(self.tipTriggers) do
                        trigger:updateFillPlane();
                    end
                    self:raiseDirtyFlags(self.husbandryDirtyFlag);
                end
                self.dirtToDrop = self.dirtToDrop + consumed;
            end
            local waterMultiplier = 0;
            if self.animalDesc.waterPerDay > 0 then
                local water = self:getFillLevel(FillUtil.FILLTYPE_WATER);
                local waterNeeded = self.totalNumAnimals * self.animalDesc.waterPerDay * dayToInterval;
                waterMultiplier = math.min(water / waterNeeded, 1);
                self.productivity = self.productivity * waterMultiplier;
                if waterNeeded > 0 and water > 0 then
                    self:changeFillLevels(-math.min(waterNeeded, water), FillUtil.FILLTYPE_WATER);
                end
            end
            -- calculate reproduction rate ...
            if self.animalDesc.birthRatePerDay > 0 and self.totalNumAnimals > 1 then
                local birthIncrease = self.productivity * self.totalNumAnimals * self.animalDesc.birthRatePerDay * dayToInterval;
                self.newAnimalPercentage = self.newAnimalPercentage + birthIncrease;
                if self.newAnimalPercentage > 1.0 then
                    local numNewAnimals = math.floor(self.newAnimalPercentage);
                    self.newAnimalPercentage = self.newAnimalPercentage - numNewAnimals;
                    self:addAnimals(numNewAnimals, 0);
                    if self.typeName == "cow" then
                        g_currentMission.missionStats:updateStats("breedCowsCount", numNewAnimals);
                    elseif self.typeName == "pig" then
                        g_currentMission.missionStats:updateStats("breedPigsCount", numNewAnimals);
                    elseif self.typeName == "sheep" then
                        g_currentMission.missionStats:updateStats("breedSheepCount", numNewAnimals);
                    end;
                end;
                -- values for stats
                self.reproductionRatePerDay = self.productivity * self.totalNumAnimals * self.animalDesc.birthRatePerDay;
            else
                self.reproductionRatePerDay = 0;
            end;
            if self.animalDesc.milkPerDay > 0 then
                local newMilk = self.productivity * numProducingAnimals * self.animalDesc.milkPerDay * dayToInterval;
                -- add the new milk
                if newMilk > 0 and self.parent.fillType[FillUtil.FILLTYPE_MILK] then
					local parentFillType = self.parent.fillType[FillUtil.FILLTYPE_MILK];
					local delta = math.min(newMilk, parentFillType.capacity - parentFillType.level);
					parentFillType.level = parentFillType.level + delta;
					if newMilk > delta then
						local text = string.format(g_i18n:getText("NoSpaceInStorage"), FillUtil.fillTypeIndexToDesc[FillUtil.FILLTYPE_MILK].nameI18N);
						g_currentMission:showBlinkingWarning(text, 3000);
					end;
                end;
            end;
            if self.liquidManureTrigger ~= nil and self.animalDesc.liquidManurePerDay > 0 and (waterMultiplier > 0 or self.animalDesc.waterPerDay == 0) then
                local newLiquidManure = waterMultiplier * self.totalNumAnimals * self.animalDesc.liquidManurePerDay * dayToInterval;
				local parentFillType = self.parent.fillType[FillUtil.FILLTYPE_LIQUIDMANURE];
				local delta = math.min(newLiquidManure, parentFillType.capacity - parentFillType.level);
                self.liquidManureTrigger:setFillLevel(parentFillType.level + delta);
				if newLiquidManure > delta then
					local text = string.format(g_i18n:getText("NoSpaceInStorage"), FillUtil.fillTypeIndexToDesc[FillUtil.FILLTYPE_LIQUIDMANURE].nameI18N);
					g_currentMission:showBlinkingWarning(text, 3000);
				end;
            end
            -- drop manure to manure area
            if self.animalDesc.manurePerDay > 0 then
                local newManure = strawMultiplier * self.totalNumAnimals * self.animalDesc.manurePerDay * dayToInterval;
                if newManure > 0 and self.parent.fillType[FillUtil.FILLTYPE_MANURE] then
					local parentFillType = self.parent.fillType[FillUtil.FILLTYPE_MANURE];
					local delta = math.min(newManure, parentFillType.capacity - parentFillType.level);
					self.manureToDrop = self.manureToDrop + delta;
					if newManure > delta then
						local text = string.format(g_i18n:getText("NoSpaceInStorage"), FillUtil.fillTypeIndexToDesc[FillUtil.FILLTYPE_MANURE].nameI18N);
						g_currentMission:showBlinkingWarning(text, 3000);
					end;
				end;
            end;
            if self.animalDesc.palletFillLevelPerDay > 0 then
                local fillDelta = self.productivity * numProducingAnimals * self.animalDesc.palletFillLevelPerDay * dayToInterval;
                if fillDelta > 0 and self.parent.fillType[self.palletFillType] then
					local parentFillType = self.parent.fillType[self.palletFillType];
					local delta = math.min(fillDelta, parentFillType.capacity - parentFillType.level);
					parentFillType.level = parentFillType.level + delta;
					if fillDelta > delta then
						local text = string.format(g_i18n:getText("NoSpaceInStorage"), FillUtil.fillTypeIndexToDesc[self.palletFillType].nameI18N);
						g_currentMission:showBlinkingWarning(text, 3000);
					end;
                end;
            end;
            if self.pickupObjectsId ~= nil and self.animalDesc.pickUpObjectsPerDay > 0 then
				self.numPickupObjectsToSpawn = self.numPickupObjectsToSpawn + self.productivity*numProducingAnimals * self.animalDesc.pickUpObjectsPerDay * dayToInterval;
                local numToSpawn = math.floor(self.numPickupObjectsToSpawn);
                self.numPickupObjectsToSpawn = self.numPickupObjectsToSpawn - numToSpawn;
                if numToSpawn > 0 then
					if UniversalFactoryHUD.settings[UniversalFactoryHUD.SET_EGGAUTOPICKUP] then
						g_currentMission:setNumPickupObjects(self.pickupObjectsFillType, g_currentMission:getNumPickupObjects(self.pickupObjectsFillType)+numToSpawn);
						g_currentMission:addSharedMoney(-2*numToSpawn, "wagePayment");
					else
						self:spawnPickupObjects(numToSpawn);
					end;
                end;
            end;
            for _,trigger in pairs(self.tipTriggers) do
                trigger:updateFillPlane();
            end;
            self:raiseDirtyFlags(self.husbandryDirtyFlag);
			self.parent:heapsMoving();
			self.parent.enableUpdateClients = true;
        end;
        if self.animalDesc.manurePerDay > 0 and self.parent.fillType[FillUtil.FILLTYPE_MANURE] then
            self:updateManureStatistics();
        end;
    end;
end;

function AnimalHusbandry:getDataAttributes()
    local attributesGeneral = {};
    local attributesSpecific = {};
    local attributesInput = {};
    local attributesHelp = {};
    local i18n = g_i18n;
    if self.customEnvironment ~= nil then
        i18n = _G[self.customEnvironment].g_i18n;
    end
    -- output genral
    table.insert(attributesGeneral, {name=i18n:getText("statistic_"..self.typeName.."Owned"), value=string.format("%d", self.totalNumAnimals)});
    table.insert(attributesGeneral, {name=g_i18n:getText("statistic_productivity"), value=string.format("%d%%", math.floor(self.productivity*100.0)), isPercentValue=true, percent=self.productivity});
    if self.reproductionRatePerDay == 0 then
        table.insert(attributesGeneral, {name=g_i18n:getText("statistic_reproductionRate"), value=string.format("--:--h")});
        table.insert(attributesGeneral, {name=g_i18n:getText("statistic_timeTillNextAnimal"), value=string.format("--:--h")});
    else
        local reproductionDuration = 1.0 / self.reproductionRatePerDay;
        local reproMins = reproductionDuration * 24 * 60;
        local hours = math.floor(reproMins / 60);
        local mins = reproMins - (hours*60);
        table.insert(attributesGeneral, {name=g_i18n:getText("statistic_reproductionRate"), value=string.format("%02d:%02dh", hours, mins )});
        local percentageMissing = math.max(0, 1.0 - self.newAnimalPercentage);
        local timeMissing = percentageMissing * reproMins;
        local hours = math.floor(timeMissing / 60);
        local mins = timeMissing - (hours*60);
        table.insert(attributesGeneral, {name=g_i18n:getText("statistic_timeTillNextAnimal"), value=string.format("%02d:%02dh", hours, mins)});
    end
    -- output specific
    if self.parent.fillType[FillUtil.FILLTYPE_LIQUIDMANURE] then
        local level = self.parent.fillType[FillUtil.FILLTYPE_LIQUIDMANURE].level;
		table.insert(attributesSpecific, {name=g_i18n:getText("statistic_liquidManureStorage").. " [" .. g_i18n:getText("unit_literShort") .."]", value=self:getFluidStatsText(level, true)});
    end
    if self.parent.fillType[FillUtil.FILLTYPE_MANURE] then
        local level = self.parent.fillType[FillUtil.FILLTYPE_MANURE].level;
		table.insert(attributesSpecific, {name=g_i18n:getText("statistic_manureStorage").. " [" .. g_i18n:getText("unit_literShort") .."]", value=self:getFluidStatsText(level, true)});
    end
    if self.parent.fillType[FillUtil.FILLTYPE_MILK] then
		local level = self.parent.fillType[FillUtil.FILLTYPE_MILK].level;
        table.insert(attributesSpecific, {name=g_i18n:getText("statistic_milkStorage").. " [" .. g_i18n:getText("unit_literShort") .."]", value=self:getFluidStatsText(level)});
    end
    if self.parent.fillType[FillUtil.FILLTYPE_WOOL] then
        local fillTypeI18N = FillUtil.fillTypeIndexToDesc[FillUtil.FILLTYPE_WOOL].nameI18N;
        if self.isServer then
            local fillLevel = self.parent.fillType[FillUtil.FILLTYPE_WOOL].level;
            local fillPercentage = math.floor(100*fillLevel / self.parent.fillType[FillUtil.FILLTYPE_WOOL].capacity);
            table.insert(attributesSpecific, {name=fillTypeI18N.. " [" .. g_i18n:getText("unit_literShort") .."]", value=string.format("%d (%d%%)", g_i18n:getFluid(fillLevel), fillPercentage), isPercentValue=true, percent=fillPercentage/100});
        else
            table.insert(attributesSpecific, {name=fillTypeI18N, value=string.format("%d%%", self.currentPalletFillPercentage)});
        end
    end
    -- input
    if self.animalDesc.dirtFillLevelPerDay > 0 then
        if self.totalNumAnimals > 0 then
            table.insert(attributesInput, {name=g_i18n:getText("statistic_cleanliness").. " [%]", value=string.format("%d%%", math.floor(self.cleanlinessFactor*100)), isPercentValue=true, percent=self.cleanlinessFactor});
        else
            table.insert(attributesInput, {name=g_i18n:getText("statistic_cleanliness").. " [%]", value=string.format("-"), isPercentValue=true, percent=0});
        end
    end
    if self.tipTriggersFillLevels[FillUtil.FILLTYPE_WATER] ~= nil then
        local waterCapacity = self:getCapacity(FillUtil.FILLTYPE_WATER)
        local percent = Utils.clamp(self:getFillLevel(FillUtil.FILLTYPE_WATER) / waterCapacity, 0, 1)
        table.insert(attributesInput, {name=g_i18n:getText("statistic_water").. " [" .. g_i18n:getText("unit_literShort") .."]", value=self:getFluidStatsText(self:getFillLevel(FillUtil.FILLTYPE_WATER)), isPercentValue=true, percent=percent});
    end
    if self.tipTriggersFillLevels[FillUtil.FILLTYPE_STRAW] ~= nil then
        local strawCapacity = self:getCapacity(FillUtil.FILLTYPE_STRAW)
        local percent = Utils.clamp(self:getFillLevel(FillUtil.FILLTYPE_STRAW) / strawCapacity, 0, 1)
        table.insert(attributesInput, {name=g_i18n:getText("statistic_strawStorage").. " [" .. g_i18n:getText("unit_literShort") .."]", value=self:getFluidStatsText(self:getFillLevel(FillUtil.FILLTYPE_STRAW)), isPercentValue=true, percent=percent});
    end
    if self.animalDesc.foodPerDay > 0 and FillUtil.foodGroups[self.animalDesc.index] ~= nil then
        for _,foodGroup in pairs(FillUtil.foodGroups[self.animalDesc.index]) do
            local fillTypesString = AnimalHusbandry:getFillTypesString(foodGroup.fillTypes);
            local sumFillLevels = self:getAvailableAmountOfFillTypes(foodGroup.fillTypes);
            local capacity = self:getCapacity(nil, foodGroup)
            local percent = Utils.clamp(sumFillLevels/capacity, 0, 1)
            table.insert(attributesInput, {name=fillTypesString.. " [" .. g_i18n:getText("unit_literShort") .."]", value=string.format("%d", sumFillLevels), isPercentValue=true, percent=percent});
        end
    end
    -- help
    local text = g_i18n:getText("statistic_animalRequirementsMain") .. "\n";
    local foodTextTable = {};
    for _,foodGroup in pairs(FillUtil.foodGroups[self.animalDesc.index]) do
        local fillTypeNames = {};
        for fillType,_ in pairs(foodGroup.fillTypes) do
            table.insert(fillTypeNames, FillUtil.fillTypeIndexToDesc[fillType].nameI18N);
        end
        local fillTypeNamesString = table.concat(fillTypeNames, ", ");
        table.insert(foodTextTable, string.format( g_i18n:getText("statistic_animalRequirementsFood"), math.floor((100*foodGroup.weight) + 0.5), tostring(foodGroup.nameI18N), fillTypeNamesString) );
    end
    local foodText = table.concat(foodTextTable, "\n");
    text = text .. foodText
    table.insert(attributesHelp, {name=text, value=""});
    return attributesGeneral, attributesSpecific, attributesInput, attributesHelp;
end;

function AnimalHusbandry:removeManure(delta)
    local used = 0;
    if self.manureToDrop >= delta then
        self.manureToDrop = self.manureToDrop - delta;
        used = delta;
    else
        self.manureToDrop = self.manureToDrop - delta;
        delta = math.abs(self.manureToDrop);
        self.manureToDrop = 0;
        self.manureToRemove = self.manureToRemove + delta;
        local manureLevel = self:getManureLevel();
        if self.manureToRemove < manureLevel then
            used = delta;
            if self.manureToRemove > TipUtil.getMinValidLiterValue(FillUtil.FILLTYPE_MANURE) then
                local xs,_,zs = getWorldTranslation(self.manureArea.start);
                local xw,_,zw = getWorldTranslation(self.manureArea.width);
                local xh,_,zh = getWorldTranslation(self.manureArea.height);
                local ux, uz = xw-xs, zw-zs;
                local vx, vz = xh-xs, zh-zs;
                local radius = Utils.vector2Length(xw-xh, zw-zh) * 0.5
                local sx = xs + 0.5*ux + 0.5*vx;
                local sz = zs + 0.5*uz + 0.5*vz;
                local sy = getTerrainHeightAtWorldPos(g_currentMission.terrainRootNode, sx,0,sz) + 10;
                local ex = xs + 0.5*ux + 0.5*vx;
                local ez = zs + 0.5*uz + 0.5*vz;
                local ey = sz;
                local dropped, _ = TipUtil.tipToGroundAroundLine(nil, -self.manureToRemove, FillUtil.FILLTYPE_MANURE, sx,sy,sz, ex,ey,ez, radius, radius, nil, false, nil);
                self.manureToRemove = math.max(self.manureToRemove + dropped, 0);
                if dropped == 0 then
                    used = 0;
                end;
            end;
        end;
		self:updateManureStatistics();
    end;
    return used;
end;
